
#include <stdint.h>
#include <math.h>
#include <avr/interrupt.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include "main.h"
#include "midi.h"


// Number of waveforms
#define NWAVES 8

// Number of LFO modes
#define NLFOMODES 6

// Number of CV modes
#define NCVMODES 5

#define BENDSCALELO (2.0)
#define BENDSCALEHI (36.0)
#define LOWCVCENTRE (59.0)
#define FULLSCALE (47.0)
#define FULLCENTRE (59.0)


#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))

// Forumula to convert MIDI note number to frequency is
// F=440 * (2^(1/12)) ^ (note - 69)  Assuming A=440Hz
// Then to convert to increment value
// increment = (2^32) * f/ (F_CPU)
//
// Crunching numerically and optimising we get

#define INCFROMNOTE(n) (2194.71 * exp(0.057762 * ((float) n) ) )

#define LFOLOWNOTE (-48) // Midi note equivalence to lowest LFO frequency (0.5Hz)
#define LFONOTERANGE (92) // Range of LFO tuning in semitones/MIDI notes (set for 0.5-100Hz)

#define MIDINOTEONFLAG 1
#define MIDINOTEOFFFLAG 2

#define FORVFLAG 0x20
#define OSC2MIDIOFFFLAG 0x40
#define V1ORV2FLAG 0x80

#define CV1INPUT 2
#define CV2INPUT 0

#define GATEIN (PIND & _BV(PD4))

#define EEPROMVERSION 1

#define EEPROMBASE 1

// Number of cycles of a 9kHz clock for a long press.
#define LONGPRESSTIME 9000

#define INITIALMIDISETTINGS MIDI_CHANNEL_OMNI

//global data
asm(".global wave"); //if want seen by others
asm(".data");
// The next line is in-theory necessary to force "wave" on to a page (256 byte) boundary
// but as "wave" is allocated at the start of ".data" it is aligned to boundary
// in any case. Including the next line appears to mess-up the allocation of
// the ".bss" section wasting memory
//
// After major changes I suggest you check you ".sym" file to see if "wave"
// is on a page boundary
//
//asm(".balign 256"); // Align to page boundary
asm("wave: .fill 512");
asm(".global serbuf");
asm( BUFASM );
asm(".global accumulator");
asm("accumulator: .fill 8");
asm(".global increment");
asm("increment: .fill 8");
asm(".global bufptr");
asm("bufptr: .fill 1");
asm(".global buflen");
asm("buflen: .fill 1");
asm(".text");


#define BUTTON0 (!(PINB & _BV(PB0)))
#define BUTTON1 (!(PINC & _BV(PC3)))
#define BUTTON2 (!(PINC & _BV(PC4)))
#define BUTTON3 (!(PINB & _BV(PB1)))


typedef struct {
	uint8_t wave1;
	uint8_t wave2;
	uint8_t lfomode;
	uint8_t cvmode;
	uint8_t gate;
} machinestate;


extern uint8_t  wave[2][256];// __attribute__ ((aligned (256)));

machinestate oldstate={255,255,255,255,255};
machinestate newstate;
uint8_t  midisignal=0;
uint8_t  midinotestate=0;
uint8_t  eepromupdateneeded=0;
extern volatile uint32_t accumulator[2];
extern volatile uint32_t increment[2];
volatile uint8_t lfooneshot=1; // if starting in "norm" mode
volatile uint8_t lforeset=1;

// MIDI storage globals
byte	lastnote;
byte	lastchannel;
float lastbend=0;
float lastlfo=LFOLOWNOTE; // Last LFO value received from MIDI (via "modulation" control)

// System state variables
float cv[2]={0};
uint8_t controlvalue[4];
int8_t ledcolumn=0;
int8_t debounce[4]={0};
uint8_t midisettings;

void setstateinfo(uint8_t w1,uint8_t w2, uint8_t lfom, uint8_t cvm, uint8_t mset) {
	controlvalue[0]=newstate.wave1=w1;
	controlvalue[1]=newstate.wave2=w2;
	controlvalue[2]=newstate.lfomode=lfom;
	controlvalue[3]=newstate.cvmode=cvm;
	midisettings=mset;
}

void setdefaults() {
	setstateinfo(0,0,1,0,INITIALMIDISETTINGS);
	newstate.gate=oldstate.gate=0;
}

void firstsetup(void) {
	MCUCR &= _BV(PUD); // Enable pull-up resistors
	DDRD|=_BV(PD5); // Output for OC0B
	DDRB|=_BV(PB3); // Output for gate
	DDRD|=_BV(PD3); // Set output on OC2B
	DDRD|=_BV(PD6); // Set output on OC0A
	
	// Set up Timer 1 as slow (4s wrap-around) clock to support
	// user interface LED flash
	TCCR1A = 0;
	TCCR1B = _BV(CS12) | _BV(CS10); // Prescale down by 1024
	
	// Set up AVCC reference
	 ADMUX = _BV(REFS0); // 5V vref
	
	// Set up free running ADC with max prescaling
	
	ADCSRA=_BV(ADPS2)|_BV(ADPS1)|_BV(ADPS0);
	ADCSRA|=_BV(ADEN);
	ADCSRA|=_BV(ADSC);
	
	// Turn off analog comparator
	
	ACSR=_BV(ACD);
	
	
}

void secondsetup(void)
{
	
	
	// Set up Timer 0 trigger accumulator increment interrupt
	// and PWM for OSC0 and OSC2 output
	
	// Set fast PWM mode
	TCCR0A |= _BV(WGM01) | _BV(WGM00);
    TCCR0B &= ~_BV(WGM02);
	
	// Do inverting PWM on pin OC0A
    // On the Arduino this is pin 6.
	TCCR0A = TCCR0A | _BV(COM0A1) | _BV(COM0A0);
   
    // Alternative for non-inverting PWM
	// TCCR0A = (TCCR0A | _BV(COM0A1)) & ~_BV(COM0A0);
	
	// Do Inverting PWM on pin OC0B
	// On the Arduino this is pin 5
	TCCR0A = TCCR0A | _BV(COM0B1) | _BV(COM0B0);
	
	// Alternative for non-inverting PWM:
	// TCCR0A = (TCCR0A | _BV(COM0B1)) & ~_BV(COM0B0);
	
	// No prescaler (p.158)
    TCCR0B = (TCCR0B & ~(_BV(CS02) | _BV(CS01))) | _BV(CS00);
	

    // Set up Timer 2 to do pulse width modulation for (log f) output

    // Use internal clock (datasheet p.160)
    ASSR &= ~( _BV(AS2));

    // Set fast PWM mode  (p.157)
    TCCR2A |= _BV(WGM21) | _BV(WGM20);
    TCCR2B &= ~_BV(WGM22);

	
	// No non-inverting PWM on OC2B
	// On the Arduino this is PIN3 (PD3)
    TCCR2A = TCCR2A | _BV(COM2B1) | _BV(COM2B0);
	
	// Alernative for inverting PWM
	// TCCR2A = (TCCR2A | _BV(COM2B1)) & ~_BV(COM2B0);
	
    // No prescaler (p.158)
    TCCR2B = (TCCR2B & ~(_BV(CS22) | _BV(CS21))) | _BV(CS20);

	
	// Set counter values for timer 0 and timer2 to dephase them
	TCNT0=0;
	TCNT2=128;

	// Set timer 1 with 8 precaling to count to OCR1A
	// Timer 1 is polled to refresh LED matrix
//	TCCR1B=_BV(WGM12)|_BV(CS11); 
//	OCR1A=2500;

    cli();
  
		TIMSK0=_BV(TOIE0);

    sei();
	
}


void setosc1(float n) {
	uint32_t tempinc;
	tempinc=INCFROMNOTE(n);
	cli();
	increment[0]=tempinc;
	sei();
	if (!(midisettings & FORVFLAG))
		OCR2B=CLAMP((n-12)*2.586,0,255); // Set Log F output
}

static inline void stoposc1() {
	cli();
	increment[0]=0;
	sei();
}

void setosc2(float n, uint8_t resetflag) {
	uint32_t tempinc;
	tempinc=INCFROMNOTE(n);
	if (! resetflag) {
		cli();
		increment[1]=tempinc;
		sei();
	} else {
		cli();
		increment[1]=tempinc;
		accumulator[1]=0;
		sei();
	}
} 

static inline void stoposc2() {
	cli();
	increment[1]=0;
	sei();
}

void HandleNoteOff(byte channel, byte pitch, byte velocity) {
	if ((channel==lastchannel) && (pitch==lastnote)) {
		midisignal|=MIDINOTEOFFFLAG;
		midinotestate =0;
	}
}

void HandleNoteOn(byte channel, byte pitch, byte velocity) {
	if (velocity==0) {
		HandleNoteOff(channel, pitch, velocity);
	} else {
		midisignal|=MIDINOTEONFLAG;
		lastchannel=channel;
		lastnote=pitch;
		midinotestate=1;

		if (midisettings & FORVFLAG)
			OCR2B=CLAMP(velocity<<1,0,255); // Set Log F output
	}
}


void PitchBendProc(byte channel, int bend) {
	if (channel==lastchannel) {
		lastbend=((float)bend)/2500.0;
	}
}

void ControlChangeProc(byte channel, byte control, byte value) {
	if (control==1) {
		lastlfo=LFOLOWNOTE+LFONOTERANGE*(float)value/127.0;
		}
	else
		if ((control>=120) && (channel==lastchannel)) {
			HandleNoteOff(channel, lastnote, 0);
		}
}


static inline void readcv(uint8_t knob, int16_t sample) {
	cv[knob]=((float)(sample-0x1ff))/((float)0x1ff);
}



void setledrow(uint8_t r) {
// Test
//	DDRD|=_BV(PD7);
//	PORTD&=~_BV(PD7);
	switch(r) {
		case 0:
			DDRD|=_BV(PD7);
			PORTD&=~_BV(PD7);
		break;
		case 1:
			DDRC|=_BV(PC1);
			PORTC&=~_BV(PC1);
		break;
		case 2:
			DDRD|=_BV(PD2);
			PORTD&=~_BV(PD2);
		break;
		case 3:
			DDRD|=_BV(PD1);
			PORTD&=~_BV(PD1);
		break;
		case 4:
			DDRC|=_BV(PC5);
			PORTC&=~_BV(PC5);
		break;
		default:;
	}
}

void setledmatrix(uint8_t itson, uint8_t ledcolumn, uint8_t ledrow) {

	DDRB&=~(_BV(PB0)|_BV(PB1)|_BV(PB2));
	DDRC&=~(_BV(PC1)|_BV(PC3)|_BV(PC4)|_BV(PC5));
	DDRD&=~(_BV(PD1)|_BV(PD2)|_BV(PD7));
	
	PORTB&=~(_BV(PB0)|_BV(PB1)|_BV(PB2));
	PORTC&=~(_BV(PC1)|_BV(PC3)|_BV(PC4)|_BV(PC5));
	PORTD&=~(_BV(PD1)|_BV(PD2)|_BV(PD7));
	
	if (itson || (ledcolumn==4)) {
		switch(ledcolumn) {
			case 0:
				DDRB|=_BV(PB0);
				PORTB|=_BV(PB0);
			break;
			case 1:
				DDRC|=_BV(PC3);
				PORTC|=_BV(PC3);
			break;
			case 2:
				DDRC|=_BV(PC4);
				PORTC|=_BV(PC4);
			break;
			case 3:
				DDRB|=_BV(PB1);
				PORTB|=_BV(PB1);
				setledrow(controlvalue[ledcolumn]);
			break;
			case 4:
				DDRB|=_BV(PB2); // Set Row 6 output low
				PORTB|=_BV(PB0)|_BV(PB1);
				PORTC|=_BV(PC3)|_BV(PC4);
			break;
			default:;
		}
	}
	if (itson && (ledcolumn !=4))
		setledrow(ledrow);
}

uint8_t wavetorow(uint8_t w) {
	if (w==0)
		return 0;
	if (w<=3)
		return 1;
	if ((w==4) || (w==5))
		return w-2;
	else
		return w-3;
}
		
uint8_t wavetoison(uint8_t w) {
	if (w==2)
		return (TCNT1 & 0x3fff) > 0x400;
	if ((w==3) || (w==6))
		return (TCNT1 & 0xfff) > 0x400;
	else
		return true;
}

uint8_t modetoison(uint8_t m) {
	if (m<=3)
		return true;
	if (m==4)
		return (TCNT1 & 0x3fff) > 0x400;
	return (TCNT1 & 0xfff) > 0x400;
}
		
int8_t processbutton(int8_t bitstate, int8_t button) {
	if (bitstate) {
		if (debounce[button]>2) {
			debounce[button]=0;
			return true;
			}
	} else {
		if (debounce[button]<100)
			debounce[button]+=1;
	}
	return false;
}

void updatematrix() {
	uint8_t controlchange=0;
	uint8_t inc=1;
	if (ledcolumn==4) {
		if (processbutton(BUTTON0,0))
			{
			  if (!(midisettings & V1ORV2FLAG)) 
			  {
			    if (controlvalue[0]==1)
			      inc=3;
			     else if (controlvalue[0]==5)
			       inc=2;
			   }
			 controlvalue[0]=newstate.wave1=(controlvalue[0]+inc)%NWAVES;
			 controlchange=1;
			}
		if (processbutton(BUTTON1,1))
			{
			  if (!(midisettings & V1ORV2FLAG))
			  {
			    if (controlvalue[1]==1)
			      inc=3;
			     else if (controlvalue[1]==5)
			       inc=2;
				}
			  controlvalue[1]=newstate.wave2=(controlvalue[1]+inc)%NWAVES; 
			  controlchange=1;
			}
		if (processbutton(BUTTON2,2))
			{controlvalue[2]=newstate.lfomode=(controlvalue[2]+1)%((midisettings&V1ORV2FLAG)?NLFOMODES:NLFOMODES-2); controlchange=1;}
		if (processbutton(BUTTON3,3))
			{controlvalue[3]=newstate.cvmode=(controlvalue[3]+1)%NCVMODES; controlchange=1;}
		if (controlchange) {
			eepromupdateneeded=true;
			TCNT1&=0x3fff;
			TIFR1=_BV(TOV1); // Clear overflow flag
		}
	}
	ledcolumn=(ledcolumn+1)%5;
	
	switch (ledcolumn) {
		case 0:
		case 1:
		setledmatrix(wavetoison(controlvalue[ledcolumn]),ledcolumn,wavetorow(controlvalue[ledcolumn]));
		break;
		case 2:
		setledmatrix(modetoison(controlvalue[ledcolumn]),ledcolumn,CLAMP(controlvalue[ledcolumn],0,3));
		break;
		case 3:
		setledmatrix(1,ledcolumn,controlvalue[ledcolumn]);
		break;
		default:
		setledmatrix(1,ledcolumn,0);

	}
	

}

void polladc(void) {
	int16_t sample;
	uint8_t lastmux;
	if (!(ADCSRA & _BV(ADSC))) {
		lastmux=ADMUX & 0xf;
		if (lastmux==2)
			ADMUX=(ADMUX & 0xf0);
		else
			ADMUX=(ADMUX & 0xf0)|2;
		sample=ADCL;
		sample|=(ADCH<<8);
		switch (lastmux) {
			case CV1INPUT:
				readcv(0, sample);
				break;
			case CV2INPUT:
				readcv(1, sample);
				break;
		} // switch
		updatematrix();
		ADCSRA|=_BV(ADSC); // start next conversion
	} // if
	
}

void dooften(void) {
	polladc();
	MIDI.read();
}

void setwave(uint8_t table[], int8_t w) {
	int16_t i;
	float cosvar=1.0;
	float sinvar=0.0;
	for (i=0;i<256;i++) {
		dooften();
		switch (w) {
			case 0:
				table[i]=255-i;
				break;
			case 1:
				if (i<=127)
					table[i]=255;
				else
					table[i]=0;
				break;
			case 2:
				if (i<=73)
					table[i]=255;
				else
					table[i]=0;
				break;
			case 3:
				if (i<=38)
					table[i]=255;
				else
					table[i]=0;
				break;
			case 4:
				table[i]=(128+127*sinvar);
				// Kludge to calculate Sine while avoiding importing the sin function which is too big
				// Method used by the Sinclair Scientific caculator
				// Effectively rotating a unit vector by 1/256th of a circle each pass
				sinvar+=cosvar*0.02455; // 0.02455 = 2 * pi/256
				cosvar-=sinvar*0.02455;
		//		table[i]=128+127*sin(2.0*3.14129*i/256.0);
				break;
			case 5:
				if (i<=127)
					table[i]=2*i;
				else
					table[i]=255-2*(i-128);
				break;
			case 6:
				if (i<=127)
					table[i]=128+i;
				else
					table[i]=255-i;
			break;
			case 7:
				table[i]=i;
				break;
		}
	}
}

static inline void gateon(void) {
	PORTB |= _BV(PB3);
}

static inline void gateoff(void) {
	PORTB &= ~ _BV(PB3);
}

float calcoscfreq(uint8_t mode, uint8_t cvindex) {
	switch (mode) {
		case 0:	return (lastbend+lastnote);
		break;
		case 1:	return (BENDSCALELO*cv[cvindex]+lastbend+lastnote);
		break;
		case 2:	return (BENDSCALEHI*cv[cvindex]+lastbend+lastnote);
		break;
		case 3: return (BENDSCALEHI*cv[cvindex]+LOWCVCENTRE);
		break;
		case 4: return (FULLSCALE*cv[cvindex]+FULLCENTRE);
		break;
		default: 
		return (LOWCVCENTRE);
	}
}


void updatesound(void) {
	uint8_t resetlfoflag=0; // Set true if we are going to force the LFO to restart
	float	osc1freq, osc2freq;
	if (midisignal & MIDINOTEONFLAG) // Start of a MIDI note?
		gateoff(); // Force external gate signal low immediately before we start a new note
		
	if ((midisignal & MIDINOTEONFLAG) || ((! oldstate.gate) && newstate.gate) ) { // Starting a new note?
		resetlfoflag |= (newstate.lfomode == 1) || (newstate.lfomode==2);
		if (newstate.lfomode==1)
			lfooneshot=0; // Make sure LFO repeats in Norm mode as it might have been stopped by the end of a note
	}
	if (newstate.wave1 != oldstate.wave1)
		setwave(wave[0],newstate.wave1);
	if (newstate.wave2 != oldstate.wave2)
		setwave(wave[1],newstate.wave2);
	if (newstate.lfomode != oldstate.lfomode) {
		if (newstate.lfomode == 2)
			lfooneshot=1;
		else {
			if ((newstate.lfomode!=1) || midinotestate || newstate.gate)
				lfooneshot=0;
			else lfooneshot=1;
			resetlfoflag=1;
		};
	};
					
	osc1freq=calcoscfreq(newstate.cvmode, 0);
	
	switch(newstate.lfomode) {
		case 3:
		osc2freq=calcoscfreq(newstate.cvmode,1);
		break;
		case 4:
		osc2freq=calcoscfreq(newstate.cvmode,1)-12.0;
		break;
		case 5:
		osc2freq=calcoscfreq(newstate.cvmode,1)+12.0;
		break;
		default:
			switch ((midisettings&OSC2MIDIOFFFLAG)?4:newstate.cvmode) {
			case 0: osc2freq=lastlfo;
			break;
			case 1:
			case 2: osc2freq=LFONOTERANGE/2.0+lastlfo+cv[1]*LFONOTERANGE/2.0;
			break;
			default:
				osc2freq=LFOLOWNOTE+LFONOTERANGE/2.0+cv[1]*LFONOTERANGE/2.0;
		}

	}

	if (midinotestate)
		gateon();
	else
		gateoff();
	
	if (midinotestate || newstate.gate) { // Note is playing
		setosc1(osc1freq);
	}
	else {
		stoposc1();
	}
	
	if (newstate.lfomode<3) {
		if ((newstate.lfomode !=2) || resetlfoflag)
			setosc2(osc2freq, resetlfoflag); // LFO mode 0 or 1 or (2 with new note)
		if ((newstate.lfomode==1) && (
				((midisignal & MIDINOTEOFFFLAG) && (!newstate.gate)) || 
				((!midinotestate) && oldstate.gate && (!newstate.gate))
				)
			)
				lfooneshot=1; // In norm mode stop oscillator at end of note
	} else { // LFO mode >= 3 - Second oscillator mode
		if (midinotestate || newstate.gate)
			setosc2(osc2freq,0);
		else
			stoposc2();
	}
}

void updateeeprom(uint8_t withpolling) {
	eepromupdateneeded=0;
	eeprom_update_byte((uint8_t*)(EEPROMBASE),EEPROMVERSION);
	if (withpolling)
		do { dooften(); } while (!eeprom_is_ready());
	eeprom_update_byte((uint8_t*)(EEPROMBASE+1),newstate.wave1);
	if (withpolling)
		do { dooften(); } while (!eeprom_is_ready());
	eeprom_update_byte((uint8_t*)(EEPROMBASE+2),newstate.wave2);
	if (withpolling)
		do { dooften(); } while (!eeprom_is_ready());
	eeprom_update_byte((uint8_t*)(EEPROMBASE+3),newstate.lfomode);
	if (withpolling)
		do { dooften(); } while (!eeprom_is_ready());
	eeprom_update_byte((uint8_t*)(EEPROMBASE+4),newstate.cvmode);
	if (withpolling)
		do { dooften(); } while (!eeprom_is_ready());
	eeprom_update_byte((uint8_t*)(EEPROMBASE+5),midisettings);
	do { 
		if (withpolling)
			dooften(); 
	} while (!eeprom_is_ready());
	EEARL=0; // Leave the address point to "sacrificial" address 0
}

uint8_t verifyvalues(uint8_t w1,uint8_t w2, uint8_t lfom, uint8_t cvm, uint8_t mset) {
    uint8_t basicok;
	basicok=(w1<NWAVES) && (w2<NWAVES) && (lfom<NLFOMODES) && (cvm < NCVMODES) && ((mset & 0x1f) <=16);
	if ((!basicok) || (mset & V1ORV2FLAG))
	  return basicok;
	return ((((1<<w1)||(1<<w2)) & 0x4c) == 0) && 
	       (lfom<NLFOMODES-2) && (mset==INITIALMIDISETTINGS); // Test that w1 and w2 don't have disallowed values for v1
}

uint8_t readeeprom() {
	uint8_t w1,w2,lm,cm,m;
	if (eeprom_read_byte((uint8_t*)(EEPROMBASE)) == EEPROMVERSION) {
		w1=eeprom_read_byte((uint8_t*)(EEPROMBASE+1));
		w2=eeprom_read_byte((uint8_t*)(EEPROMBASE+2));
		lm=eeprom_read_byte((uint8_t*)(EEPROMBASE+3));
		cm=eeprom_read_byte((uint8_t*)(EEPROMBASE+4));
		m=eeprom_read_byte((uint8_t*)(EEPROMBASE+5));
		if (verifyvalues(w1,w2,lm,cm,m)) {
			setstateinfo(w1,w2,lm,cm,m);
			EEARL=0; // Leave the address point to "sacrificial" address 0
			return 1;
		}
	}
	return 0;
	
}



void poweron(uint8_t scan) {
	uint8_t row=0;
	uint8_t midi;
	uint8_t step=0;
	TCNT1=0;
	if (scan) {
	   while(step<4) {
	      	while (ADCSRA & _BV(ADSC)) // Using the ADC as a crude timer
			{};
		row=(row+1)%7;
		if (row<4 && (step &1) ==0)
		   setledmatrix(1, 1+(row&1), 1+((row&2)>>1));
		else
		   setledmatrix(0,0,0);
		if (TCNT1>0x3000) {
           step+=1;
           TCNT1=0;
          }  
        ADCSRA|=_BV(ADSC); 
	   }
	}
	if (midisettings & V1ORV2FLAG) {
	  row=0;
	  midi=midisettings & 0x1f;
	  if (midi == MIDI_CHANNEL_OMNI)
		  midi=0x1f;
	  TCNT1=0;
	
	  while (TCNT1<=0x6000) {
		while (ADCSRA & _BV(ADSC)) // Using the ADC as a crude timer
			{};
		row=(row+1)%7;
		if (row<5)
			setledmatrix(midi & (1<<row), 1, row);
		else
			if (row==5)
				setledmatrix(1,2,(midisettings&FORVFLAG)?1:0);
			else
				setledmatrix(1,2,(midisettings&OSC2MIDIOFFFLAG)?3:2);
		ADCSRA|=_BV(ADSC);
	  }
	}
	

}

void poweronwithbinselect() {
	uint8_t row=0;
	uint8_t flagmask;
	
	flagmask = (midisettings >> 5) & 3;
	while(true) {
		while (ADCSRA & _BV(ADSC)) // Using the ADC as a crude timer. Approx 9kHz.
			{};
		if (row==5) {
			if (BUTTON0 || BUTTON1 || BUTTON3)
				break;
			if (processbutton(BUTTON2,2))
				flagmask=(flagmask+1)&3;
		} // row ==5
		row=(row+1)%6;
		if (row<2)
			setledmatrix(1,2,(row*2)+((flagmask & (1+row))?1:0));
		else
			setledmatrix(0,4,0);
		ADCSRA|=_BV(ADSC);
	}
	midisettings=(midisettings & 0x9f) | (flagmask << 5);
	updateeeprom(0);
	poweron(0);
}

void poweronwithv1v2select() {
   uint8_t row=0;
   uint8_t rowcount=4;
   uint16_t rowtimer=0;
   while (true) {
   		while (ADCSRA & _BV(ADSC)) // Using the ADC as a crude timer. Approx 9kHz.
			{};
		if (row==5 && (! BUTTON0))
		{
		   setdefaults();
		   break;
		}
		if (rowtimer++>LONGPRESSTIME/5) {
		  rowtimer=0;
		  if (rowcount==0)
		  {
		    setdefaults();
			midisettings |= V1ORV2FLAG;
			break;
		   }
		   else
		    rowcount--;
		}
		row=(row+1)%6;
		if (row!=5)
			setledmatrix(row==rowcount, 0, row);
		else
			setledmatrix(0, 4, 0);

		ADCSRA|=_BV(ADSC);
   }
   updateeeprom(0);
   poweron(0);
}

void poweronwithmidichanselect() {
	uint8_t row=0;
	uint8_t midi;
	
	
	midi=midisettings & 0x1f;
	if (midi == MIDI_CHANNEL_OMNI)
		midi=0x1f;
	
	while (true) {
		while (ADCSRA & _BV(ADSC)) // Using the ADC as a crude timer. Approx 9kHz.
			{};
		if (row==5 && (BUTTON0 || BUTTON2 || BUTTON3)) // Quit if any other button pressed
			break;
		if (row==5) {
			if (processbutton(BUTTON1,1)) {
				if (midi==0x1f)
					midi=1;
				else
					if (midi!=16)
						midi+=1;
					else
						midi=0x1f;
			}
		}
			
		row=(row+1)%6;
		if (row!=5)
			setledmatrix(midi & (1<<row), 1, row);
		else
			setledmatrix(0, 4, 0);

		ADCSRA|=_BV(ADSC);
	}
	

	if (midi==0x1f)
		midi=MIDI_CHANNEL_OMNI;
	midisettings=(midisettings&0xe0)|midi;

	updateeeprom(0);
	poweron(0);
}
/*
void poweronerror() {
	uint8_t row=0;
	while(true) {
		while (ADCSRA & _BV(ADSC)) // Using the ADC as a crude timer. Approx 9kHz.
			{};
		row=(row+1)%5;
		setledmatrix(1,3,row);
		ADCSRA|=_BV(ADSC);
	}
}
*/

int main(void){
	firstsetup();
	// Setup to read buttongs	
	setledmatrix(0,4,0);
	_delay_ms(20);
	if (!readeeprom())
		{
		   setdefaults();
		   updateeeprom(0);
		}
	if (BUTTON0)
		poweronwithv1v2select();
	else
		if (BUTTON1 && (midisettings & V1ORV2FLAG))
			poweronwithmidichanselect();
		else
			if (BUTTON2 && (midisettings & V1ORV2FLAG))
				poweronwithbinselect();
			else
				poweron(1);
	setwave(wave[0],newstate.wave1);
	setwave(wave[1],newstate.wave2);
	secondsetup();
	MIDI.begin(midisettings&0x1f);
	MIDI.setHandleNoteOn(HandleNoteOn);
	MIDI.setHandleNoteOff(HandleNoteOff);
	MIDI.setHandlePitchBend(PitchBendProc);
	MIDI.setHandleControlChange(ControlChangeProc);

	
	while (1) {
		polladc();
		MIDI.read();
		newstate.gate=GATEIN;
		updatesound();
		if (eepromupdateneeded && (TIFR1 & _BV(TOV1)))
			updateeeprom(1);
		oldstate=newstate;
		midisignal=0;
	}

	
}
